import '@babel/polyfill';
import $ from 'jquery';
import 'es6-promise';
import 'isomorphic-fetch';
import NavigatedViewer from 'bpmn-js/lib/NavigatedViewer';
import TooltipsModule from 'diagram-js/lib/features/tooltips';

import './utils/polyfill-ie11-nodelist-foreach';
import CXDiagramModel from './model/CXDiagramModel';
import CXPIModel from './model/CXPIModel';
import BPMNBuilder from './BPMNBuilder';
import ProcessStatusRendererModule from './renderer/ProcessStatusRenderer';
import ProcessStatusColorProviderModule from './provider/ProcessStatusColorProvider';
import ZoomControlModule from './features/ZoomControl';
import PrintControlModule from './features/PrintControl';
import PlaybackControlModule from './features/PlaybackControl';
import { globalResource } from './model/ResourceModel';
import DiagramViewer from './DiagramViewer';
import ProcessStatusTooltipProvider from './provider/ProcessStatusTooltipProvider';

export default class ProcessStatusViewer extends DiagramViewer {
  constructor(objAttrs, params, embedAttrs) {
    super(objAttrs, params, embedAttrs);

    Promise.all([
      this.loadPI(globalResource.get('PIID')),
      this.loadPT(globalResource.get('ptName')),
    ]).then(
      function success() {
        console.log('Load Success');
        this.loadData();
        this.draw(globalResource.get('themeName'));
      }.bind(this)
    );
  }

  setViewerContainer(selector) {
    $(selector).empty();
    $(selector).css({
      height: '100%',
      display: 'flex',
      flexDirection: 'column',
    });
    this.headerCont = $('<div name="legend"/>').appendTo(selector);
    this.diagramCont = $('<div name="diagram"/>')
      .css({ flex: 1 })
      .appendTo(selector);
    this.viewer = new NavigatedViewer({
      container: this.diagramCont,
      additionalModules: [
        ProcessStatusColorProviderModule,
        ProcessStatusTooltipProvider,
        ProcessStatusRendererModule,
        ZoomControlModule,
        PrintControlModule,
        PlaybackControlModule,
        TooltipsModule,
      ],
    });

    const eventBus = this.viewer.get('eventBus');
    eventBus.on('element.click', 1000, this.onElementClick);
  }

  loadData() {
    const eventBus = this.viewer.get('eventBus');
    this.model = new CXDiagramModel();
    this.model.load(this.ptXML, this.piXML);

    this.piModel = new CXPIModel();
    this.piModel.load(this.piXML);
    eventBus.fire('pi.load.complete', { piModel: this.piModel });

    const builder = new BPMNBuilder();
    this.bpmnXML = builder.build(this.model);
    this.bpmnModel = builder.getBPMNModel();
    console.log(this.bpmnXML);

    const piId = globalResource.get('PIID');
    const eventUrl = `${this.psvEventsUrl}?PIID=${piId}`;
    eventBus.fire('psv.event.load', { eventXMLLocation: eventUrl });
  }

  draw(themeName) {
    const eventBus = this.viewer.get('eventBus');

    eventBus.fire('psv.legend.init', {
      headerCont: this.headerCont,
      legendData: this.model.getLegendData(),
    });
    eventBus.fire('psv.control.init', { themeName, headerCont: this.headerCont });

    this.viewer.importXML(this.bpmnXML, () => {
      ProcessStatusRendererModule.renderNotesOverlay(this.viewer, this.bpmnModel);
    });
    eventBus.fire('cxmodel.loaded', this.bpmnModel);
  }

  loadPI(piId) {
    const url = `${this.piURL}?PIID=${piId}`;

    return fetch(url).then(response =>
      response.text().then(responseXML => this.onPILoadComplete(responseXML))
    );
  }

  onPILoadComplete(xmlData) {
    this.piXML = xmlData;
  }

  onElementClick(event) {
    const activityName = event.element.businessObject.get('cx:workstepName');
    const elementType = event.element.businessObject.get('cx:cxType');

    if (activityName) {
      const piId = globalResource.get('PIID');
      const ptId = globalResource.get('PTID');

      if (elementType === 'NestedWS') {
        const args = [];

        args.push(`wsName=${activityName}`);
        args.push(`PIID=${piId}`);

        showSubprocessList(args.join('&'));
      } else {
        const args = [];
        args.push(`PTID=${ptId}`);
        args.push(`PIID=${piId}`);
        args.push(`name=${activityName}`);

        showWorkstepDetails(args.join('&'));
      }
    }
  }
}
